require("mod-gui")

if not global.where_is_it_made then
  global.where_is_it_made = {}
end

local function isempty(s)
  return s == nil or s == ''
end

function hide_gui(player)
  if player.gui.left.where_is_it_made then
    player.clear_gui_arrow()
    player.gui.left.where_is_it_made.destroy()
  end
end

function show_hide_gui(player)
  if isempty(player.name) then
    player.print({"", "show_hide_gui: player.name was empty! Please report a bug."})
    return
  end
  if not global.where_is_it_made then
    global.where_is_it_made = {}
  end
  if not global.where_is_it_made[player.name] then
    global.where_is_it_made[player.name] = {
      selected_recipe = nil,
      selected_item = nil,
      last_found_recipe_entity = nil,
      last_found_item_entity = nil
    }
  end

  if player.gui.left.where_is_it_made then
    hide_gui(player)
  else
    local frame = player.gui.left.add{
      type = "frame",
      name = "where_is_it_made",
      direction = "vertical",
      caption = {"where_is_it_made.title"}
    }
    local table = frame.add{
      type = "table",
      name = "table",
      column_count = 2
    }
    table.add{
      type = "choose-elem-button",
      name = "recipe_chooser",
      elem_type = "recipe",
      recipe = global.where_is_it_made[player.name].selected_recipe,
      tooltip = {"where_is_it_made.recipe_chooser_tooltip"}
    }
    table.add{
      type = "sprite-button",
      name = "where_is_it_made_find_recipe",
      sprite = "item/assembling-machine-3",
      tooltip = {"where_is_it_made.find_recipe_tooltip"}
    }
    table.add{
      type = "choose-elem-button",
      name = "item_chooser",
      elem_type = "item",
      item = global.where_is_it_made[player.name].selected_item,
      tooltip = {"where_is_it_made.item_chooser_tooltip"}
    }
    table.add{
      type = "sprite-button",
      name = "where_is_it_made_find_item",
      sprite = "item-group/intermediate-products",
      tooltip = {"where_is_it_made.find_item_tooltip"}
    }
    frame.add{
      type = "button",
      name = "where_is_it_made_close",
      caption= {"where_is_it_made.close"},
      tooltip = {"where_is_it_made.close_tooltip"}
    }
  end
end

function highlight_next_recipe(player)
  if player.gui.left.where_is_it_made then
    local chooser = player.gui.left.where_is_it_made.table.recipe_chooser
    local chosen_recipe_name = chooser.elem_value
    if not chosen_recipe_name then
      return
    end
    if isempty(player.name) then
      player.print({"", "highlight_next_recipe: player.name was empty! Please report a bug."})
      return
    end
    if not global.where_is_it_made then
      global.where_is_it_made = {}
    end
    global.where_is_it_made[player.name].selected_recipe = chosen_recipe_name
    local entities = player.surface.find_entities_filtered{
      type = "assembling-machine",
      force = "player"
    }

    local found = false
    if entities then
      -- find the previous entity in the list so we can skip past it (on later
      -- invocations)
      local last_found_entity_index = -1
      for i, e in pairs(entities) do
        if global.where_is_it_made[player.name].last_found_recipe_entity == e then
          last_found_entity_index = i
          break
        end
      end

      -- search the list for all matching entities after the previous one
      for i, e in pairs(entities) do
        if i > last_found_entity_index then
          local recipe = e.get_recipe()
          if recipe then
            if recipe.name == chosen_recipe_name then
              found = true
              -- if we're at the end, clear the last found entity so we can loop
              -- back through the list again
              if i < #entities then
                global.where_is_it_made[player.name].last_found_recipe_entity = e
              else
                global.where_is_it_made[player.name].last_found_recipe_entity = nil
              end
              if player.render_mode == defines.render_mode.game then
                player.set_gui_arrow{
                  type = "entity",
                  entity = e
                }
              else
                player.clear_gui_arrow()
                player.zoom_to_world(e.position)
              end
              break
            end
          end
        end
      end
    end

    if not found then
      player.print{
        "where_is_it_made.recipe_not_found",
        game.recipe_prototypes[chosen_recipe_name].localised_name
      }
      player.clear_gui_arrow()
      global.where_is_it_made[player.name].last_found_recipe_entity = nil
    end
  end
end

function highlight_next_item(player)
  if player.gui.left.where_is_it_made then
    local chooser = player.gui.left.where_is_it_made.table.item_chooser
    local chosen_item_name = chooser.elem_value
    if not chosen_item_name then
      return
    end
    if isempty(player.name) then
      player.print({"", "highlight_next_item: player.name was empty! Please report a bug."})
      return
    end
    if not global.where_is_it_made then
      global.where_is_it_made = {}
    end
    global.where_is_it_made[player.name].selected_item = chosen_item_name
    local entities = player.surface.find_entities_filtered{
      type = {"container", "logistic-container"},
      force = "player"
    }

    local found = false
    if entities then
      -- find the previous entity in the list so we can skip past it (on later
      -- invocations)
      local last_found_entity_index = -1
      for i, e in pairs(entities) do
        if global.where_is_it_made[player.name].last_found_item_entity == e then
          last_found_entity_index = i
          break
        end
      end

      -- search the list for all matching entities after the previous one
      for i, e in pairs(entities) do
        if i > last_found_entity_index then
          local inventory = e.get_inventory(defines.inventory.chest)
          if inventory then
            if inventory.get_item_count(chosen_item_name) > 0 then
              found = true
              -- if we're at the end, clear the last found entity so we can loop
              -- back through the list again
              if i < #entities then
                global.where_is_it_made[player.name].last_found_item_entity = e
              else
                global.where_is_it_made[player.name].last_found_item_entity = nil
              end
              if player.render_mode == defines.render_mode.game then
                player.set_gui_arrow{
                  type = "entity",
                  entity = e
                }
              else
                player.clear_gui_arrow()
                player.zoom_to_world(e.position)
              end
              break
            end
          end
        end
      end
    end

    if not found then
      player.print{
        "where_is_it_made.item_not_found",
        game.item_prototypes[chosen_item_name].localised_name
      }
      player.clear_gui_arrow()
      global.where_is_it_made[player.name].last_found_item_entity = nil
    end
  end
end

-- toggle the UI on/off with the hotkey
script.on_event("where_is_it_made_toggle", function(event)
  show_hide_gui(game.players[event.player_index])
end)

script.on_event(defines.events.on_gui_click, function(event)
  if event.element.name == "where_is_it_made_close" then
    -- Hide the UI
    hide_gui(game.players[event.player_index])
  elseif event.element.name == "where_is_it_made_find_recipe" then
    -- Indicate the next recipe found in a machine
    highlight_next_recipe(game.players[event.player_index])
  elseif event.element.name == "where_is_it_made_find_item" then
    -- Indicate the next item found in a container
    highlight_next_item(game.players[event.player_index])
  end
end)

script.on_event(defines.events.on_gui_opened, function(event)
  if event.gui_type == defines.gui_type.custom and event.element.name == "where_is_it_made" then
    return
  end
  -- Hide the UI (unless we are opening this mod's UI)
  hide_gui(game.players[event.player_index])
end)
